import 'package:dartz/dartz.dart';

import '../../../../core/errors/failures.dart';
import '../entities/kyc_level_entity.dart';
import '../entities/kyc_application_entity.dart';

abstract class KycRepository {
  /// Get all available KYC levels
  Future<Either<Failure, List<KycLevelEntity>>> getKycLevels();

  /// Get specific KYC level by ID
  Future<Either<Failure, KycLevelEntity>> getKycLevelById(String levelId);

  /// Get current user's KYC applications
  Future<Either<Failure, List<KycApplicationEntity>>> getKycApplications();

  /// Get specific KYC application by ID
  Future<Either<Failure, KycApplicationEntity>> getKycApplicationById(
      String applicationId);

  /// Submit new KYC application
  Future<Either<Failure, KycApplicationEntity>> submitKycApplication({
    required String levelId,
    required Map<String, dynamic> fields,
  });

  /// Update existing KYC application
  Future<Either<Failure, KycApplicationEntity>> updateKycApplication({
    required String applicationId,
    required Map<String, dynamic> fields,
  });

  /// Upload document for KYC
  Future<Either<Failure, String>> uploadKycDocument({
    required String filePath,
    required String fieldId,
    String? applicationId,
  });

  /// Get current KYC status for user
  Future<Either<Failure, KycStatusEntity>> getKycStatus();
}

class KycStatusEntity {
  final bool isKycRequired;
  final bool isKycCompleted;
  final String? currentLevel;
  final int? levelNumber;
  final List<String>? availableFeatures;
  final KycApplicationEntity? activeApplication;

  const KycStatusEntity({
    required this.isKycRequired,
    required this.isKycCompleted,
    this.currentLevel,
    this.levelNumber,
    this.availableFeatures,
    this.activeApplication,
  });
}
